import { useState } from "react";
import { Link, useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { Menu, Phone, Mail, MapPin } from "lucide-react";

export default function Header() {
  const [location] = useLocation();
  const [isOpen, setIsOpen] = useState(false);

  const navigation = [
    { name: "Home", href: "/", current: location === "/", disabled: false },
    { name: "About Us", href: "/about", current: location === "/about", disabled: false },
    { name: "Services", href: "/services", current: location === "/services", disabled: false },
    { name: "Join The Team", href: "/join-team", current: location === "/join-team", disabled: false },
    { name: "Get in Touch", href: "/contact", current: location === "/contact", disabled: false },
  ];

  return (
    <div className="sticky top-0 z-50">
      {/* Contact Details Strip */}
      <div className="text-white py-2 bg-[#4dd7dc]">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex flex-col sm:flex-row justify-between items-center space-y-2 sm:space-y-0 bg-[#4dd7dc] text-[#3c59c2]">
            <div className="flex flex-col sm:flex-row items-center space-y-1 sm:space-y-0 sm:space-x-6 text-sm">
              <div className="flex items-center" data-testid="contact-phone">
                <Phone className="w-4 h-4 mr-2" />
                <span>+44 7832 920119</span>
              </div>
              <div className="flex items-center" data-testid="contact-email">
                <Mail className="w-4 h-4 mr-2" />
                <span>info@smartgadliving.co.uk</span>
              </div>
            </div>
            <div className="flex items-center text-sm" data-testid="contact-address">
              <MapPin className="w-4 h-4 mr-2" />
              <span>92 Lime Grove, Portsmouth, England, PO6 4DQ</span>
            </div>
          </div>
        </div>
      </div>
      {/* Main Header */}
      <header className="shadow-sm bg-[#3c59c2] text-[#4dd7dc]">
        <nav className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            {/* Logo */}
            <Link href="/" className="flex items-center" data-testid="link-home-logo">
              <div className="flex items-center">
                <img 
                  src="/smartgad-logo.jpg" 
                  alt="SmartGad Living Logo" 
                  className="w-8 h-8 rounded-full mr-2 object-cover"
                />
                <span className="text-lg font-semibold text-primary-500">SmartGad Living</span>
              </div>
            </Link>

            {/* Desktop Navigation */}
            <div className="hidden md:flex items-center space-x-8">
              {navigation.map((item) => (
                item.disabled ? (
                  <span
                    key={item.name}
                    className="font-medium cursor-not-allowed text-[#4dd7dc]"
                    data-testid={`link-nav-${item.name.toLowerCase().replace(/\s+/g, '-')}`}
                  >
                    {item.name}
                  </span>
                ) : (
                  <Link
                    key={item.name}
                    href={item.href}
                    className="font-medium transition-colors hover:text-slate-900 text-[#4dd7dc]"
                    data-testid={`link-nav-${item.name.toLowerCase().replace(/\s+/g, '-')}`}
                  >
                    {item.name}
                  </Link>
                )
              ))}
              
              {/* CTA Button */}
              <Button
                variant="default"
                className="bg-[#fa914a] text-white px-6 py-2 rounded-full font-medium"
                data-testid="button-free-consultation"
                onClick={() => {
                  // Navigate to contact page
                  window.location.href = '/contact';
                }}
              >
                Free Consultation
              </Button>
            </div>

            {/* Mobile menu button */}
            <div className="md:hidden">
              <Sheet open={isOpen} onOpenChange={setIsOpen}>
                <SheetTrigger asChild>
                  <Button variant="ghost" size="icon" data-testid="button-mobile-menu">
                    <Menu className="h-6 w-6" />
                  </Button>
                </SheetTrigger>
                <SheetContent side="right" className="w-80">
                  <div className="flex flex-col space-y-4 mt-8">
                    {navigation.map((item) => (
                      item.disabled ? (
                        <span
                          key={item.name}
                          className="text-lg font-medium text-gray-400 cursor-not-allowed"
                          data-testid={`link-mobile-nav-${item.name.toLowerCase().replace(/\s+/g, '-')}`}
                        >
                          {item.name}
                        </span>
                      ) : (
                        <Link
                          key={item.name}
                          href={item.href}
                          className={`text-lg font-medium transition-colors ${
                            item.current
                              ? "text-primary-600"
                              : "text-slate-600 hover:text-primary-600"
                          }`}
                          onClick={() => setIsOpen(false)}
                          data-testid={`link-mobile-nav-${item.name.toLowerCase().replace(/\s+/g, '-')}`}
                        >
                          {item.name}
                        </Link>
                      )
                    ))}
                    <div className="pt-4">
                      <Button
                        variant="default"
                        className="bg-[#fa914a] text-white px-6 py-2 rounded-full font-medium w-full"
                        data-testid="button-mobile-free-consultation"
                        onClick={() => {
                          // Navigate to contact page and close mobile menu
                          setIsOpen(false);
                          window.location.href = '/contact';
                        }}
                      >
                        Free Consultation
                      </Button>
                    </div>
                  </div>
                </SheetContent>
              </Sheet>
            </div>
          </div>
        </nav>
      </header>
    </div>
  );
}
