import { Helmet } from "react-helmet-async";
import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useMutation } from "@tanstack/react-query";
import { z } from "zod";
import { MapPin, Phone, Mail, Clock, CheckCircle } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import ConsultationCTA from "@/components/ui/consultation-cta";

const contactSchema = z.object({
  firstName: z.string().min(2, "First name must be at least 2 characters"),
  lastName: z.string().min(2, "Last name must be at least 2 characters"),
  email: z.string().email("Invalid email address"),
  phone: z.string().optional(),
  profession: z.string().optional(),
  message: z.string().min(10, "Message must be at least 10 characters"),
});

const consultationSchema = z.object({
  firstName: z.string().min(2, "First name must be at least 2 characters"),
  lastName: z.string().min(2, "Last name must be at least 2 characters"),
  email: z.string().email("Invalid email address"),
  phone: z.string().min(10, "Phone number is required for consultation"),
  profession: z.string().min(1, "Please select your profession"),
  preferredDate: z.string().min(1, "Please select a preferred date"),
  preferredTime: z.string().min(1, "Please select a preferred time"),
  notes: z.string().optional(),
});

type ContactForm = z.infer<typeof contactSchema>;
type ConsultationForm = z.infer<typeof consultationSchema>;

export default function Contact() {
  const { toast } = useToast();
  const [activeForm, setActiveForm] = useState<"contact" | "consultation">("contact");
  const [isSubmitted, setIsSubmitted] = useState(false);

  const contactForm = useForm<ContactForm>({
    resolver: zodResolver(contactSchema),
    defaultValues: {
      firstName: "",
      lastName: "",
      email: "",
      phone: "",
      profession: "",
      message: "",
    },
  });

  const consultationForm = useForm<ConsultationForm>({
    resolver: zodResolver(consultationSchema),
    defaultValues: {
      firstName: "",
      lastName: "",
      email: "",
      phone: "",
      profession: "",
      preferredDate: "",
      preferredTime: "",
      notes: "",
    },
  });

  const submitContact = useMutation({
    mutationFn: async (data: ContactForm) => {
      return await apiRequest("POST", "/api/contact", {
        ...data,
        inquiryType: "general",
      });
    },
    onSuccess: () => {
      setIsSubmitted(true);
      toast({
        title: "Message Sent",
        description: "Thank you for your message. We'll respond within 24 hours.",
      });
    },
    onError: () => {
      toast({
        title: "Failed to Send",
        description: "Please try again or contact us directly.",
        variant: "destructive",
      });
    },
  });

  const submitConsultation = useMutation({
    mutationFn: async (data: ConsultationForm) => {
      return await apiRequest("POST", "/api/consultation", data);
    },
    onSuccess: () => {
      setIsSubmitted(true);
      toast({
        title: "Consultation Booked",
        description: "We'll contact you to confirm your consultation appointment.",
      });
    },
    onError: () => {
      toast({
        title: "Booking Failed",
        description: "Please try again or call us directly.",
        variant: "destructive",
      });
    },
  });

  const contactInfo = [
    {
      icon: MapPin,
      title: "Office Address",
      content: (
        <>
          SmartGad Living Ltd<br />
          92 Lime Grove<br />
          Portsmouth, England<br />
          PO6 4DQ
        </>
      ),
      color: "bg-primary-100 text-primary-600",
    },
    {
      icon: Phone,
      title: "Phone Numbers",
      content: (
        <>
          Main: +44 7832 920119
        </>
      ),
      color: "bg-accent-100 text-accent-600",
    },
    {
      icon: Mail,
      title: "Email Addresses",
      content: (
        <>
          General: info@smartgadliving.co.uk<br />
          Careers: careers@smartgadliving.co.uk
        </>
      ),
      color: "bg-primary-100 text-primary-600",
    },
    {
      icon: Clock,
      title: "Operating Hours",
      content: (
        <>
          Monday - Friday: 8:00 AM - 6:00 PM<br />
          Saturday: 9:00 AM - 2:00 PM<br />
          Sunday: Emergency calls only
        </>
      ),
      color: "bg-accent-100 text-accent-600",
    },
  ];

  if (isSubmitted) {
    return (
      <>
        <Helmet>
          <title>Thank You - SmartGad Living</title>
        </Helmet>
        <div className="min-h-screen bg-slate-50 flex items-center justify-center py-20">
          <div className="max-w-md mx-auto text-center bg-white p-8 rounded-xl shadow-lg">
            <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-6">
              <CheckCircle className="text-green-600 w-8 h-8" />
            </div>
            <h1 className="text-2xl font-bold text-[#3c59c2] mb-4">
              {activeForm === "consultation" ? "Consultation Booked!" : "Message Sent!"}
            </h1>
            <p className="text-slate-600 mb-6">
              {activeForm === "consultation" 
                ? "We'll contact you within 24 hours to confirm your consultation appointment."
                : "Thank you for your message. We'll respond within 24 hours."
              }
            </p>
            <Button
              onClick={() => window.location.href = "/"}
              className="bg-primary-600 hover:bg-primary-700"
              data-testid="button-return-home"
            >
              Return to Home
            </Button>
          </div>
        </div>
      </>
    );
  }

  return (
    <>
      <Helmet>
        <title>Get in Touch - SmartGad Living | Contact Healthcare Recruitment Specialists</title>
        <meta name="description" content="Contact SmartGad Living for healthcare recruitment services. Book a free consultation or send us a message. Available Monday-Friday in Manchester, UK." />
      </Helmet>

      {/* Hero Section */}
      <section className="relative bg-gradient-to-br from-primary-50 to-slate-100 py-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <h1 className="text-4xl lg:text-5xl font-bold text-[#3c59c2] leading-tight mb-6">
              Get in <span className="text-primary-600">Touch</span>
            </h1>
            <p className="text-xl text-slate-600 max-w-3xl mx-auto">
              Ready to take the next step in your healthcare career? Contact us today for a free consultation or send us a message.
            </p>
          </div>
        </div>
      </section>

      <section id="contact" className="py-20 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-2 gap-16">
            {/* Contact Forms */}
            <div>
              {/* Form Tabs */}
              <div className="flex mb-8 bg-slate-100 rounded-lg p-1">
                <button
                  onClick={() => setActiveForm("contact")}
                  className={`flex-1 py-3 px-4 rounded-md font-medium transition-colors ${
                    activeForm === "contact"
                      ? "bg-white text-primary-600 shadow-sm"
                      : "text-slate-600 hover:text-slate-900"
                  }`}
                  data-testid="tab-contact"
                >
                  Send Message
                </button>
                <button
                  onClick={() => setActiveForm("consultation")}
                  className={`flex-1 py-3 px-4 rounded-md font-medium transition-colors ${
                    activeForm === "consultation"
                      ? "bg-white text-primary-600 shadow-sm"
                      : "text-slate-600 hover:text-slate-900"
                  }`}
                  data-testid="tab-consultation"
                >
                  Book Consultation
                </button>
              </div>

              {/* Contact Form */}
              {activeForm === "contact" && (
                <div>
                  <h3 className="text-2xl font-bold text-slate-900 mb-6">Send Us a Message</h3>
                  <Form {...contactForm}>
                    <form onSubmit={contactForm.handleSubmit((data) => submitContact.mutate(data))} className="space-y-6">
                      <div className="grid md:grid-cols-2 gap-6">
                        <FormField
                          control={contactForm.control}
                          name="firstName"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>First Name</FormLabel>
                              <FormControl>
                                <Input {...field} data-testid="input-contact-first-name" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={contactForm.control}
                          name="lastName"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Last Name</FormLabel>
                              <FormControl>
                                <Input {...field} data-testid="input-contact-last-name" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <FormField
                        control={contactForm.control}
                        name="email"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Email Address</FormLabel>
                            <FormControl>
                              <Input type="email" {...field} data-testid="input-contact-email" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <div className="grid md:grid-cols-2 gap-6">
                        <FormField
                          control={contactForm.control}
                          name="phone"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Phone Number (Optional)</FormLabel>
                              <FormControl>
                                <Input {...field} data-testid="input-contact-phone" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={contactForm.control}
                          name="profession"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Professional Role (Optional)</FormLabel>
                              <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                  <SelectTrigger data-testid="select-contact-profession">
                                    <SelectValue placeholder="Select your role" />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  <SelectItem value="registered-nurse">Registered Nurse</SelectItem>
                                  <SelectItem value="care-worker">Care Worker</SelectItem>
                                  <SelectItem value="healthcare-assistant">Healthcare Assistant</SelectItem>
                                  <SelectItem value="other">Other</SelectItem>
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <FormField
                        control={contactForm.control}
                        name="message"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Message</FormLabel>
                            <FormControl>
                              <Textarea
                                {...field}
                                rows={4}
                                placeholder="Tell us how we can help you..."
                                data-testid="textarea-contact-message"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <Button
                        type="submit"
                        className="w-full bg-primary-600 hover:bg-primary-700 text-white px-8 py-4 font-semibold"
                        disabled={submitContact.isPending}
                        data-testid="button-send-message"
                      >
                        {submitContact.isPending ? "Sending..." : "Send Message"}
                      </Button>
                    </form>
                  </Form>
                </div>
              )}

              {/* Consultation Form */}
              {activeForm === "consultation" && (
                <div>
                  <h3 className="text-2xl font-bold text-slate-900 mb-6">Book a Free Consultation</h3>
                  <Form {...consultationForm}>
                    <form onSubmit={consultationForm.handleSubmit((data) => submitConsultation.mutate(data))} className="space-y-6">
                      <div className="grid md:grid-cols-2 gap-6">
                        <FormField
                          control={consultationForm.control}
                          name="firstName"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>First Name</FormLabel>
                              <FormControl>
                                <Input {...field} data-testid="input-consultation-first-name" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={consultationForm.control}
                          name="lastName"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Last Name</FormLabel>
                              <FormControl>
                                <Input {...field} data-testid="input-consultation-last-name" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <div className="grid md:grid-cols-2 gap-6">
                        <FormField
                          control={consultationForm.control}
                          name="email"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Email Address</FormLabel>
                              <FormControl>
                                <Input type="email" {...field} data-testid="input-consultation-email" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={consultationForm.control}
                          name="phone"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Phone Number</FormLabel>
                              <FormControl>
                                <Input {...field} data-testid="input-consultation-phone" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <FormField
                        control={consultationForm.control}
                        name="profession"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Professional Role</FormLabel>
                            <Select onValueChange={field.onChange} defaultValue={field.value}>
                              <FormControl>
                                <SelectTrigger data-testid="select-consultation-profession">
                                  <SelectValue placeholder="Select your role" />
                                </SelectTrigger>
                              </FormControl>
                              <SelectContent>
                                <SelectItem value="registered-nurse">Registered Nurse</SelectItem>
                                <SelectItem value="care-worker">Care Worker</SelectItem>
                                <SelectItem value="healthcare-assistant">Healthcare Assistant</SelectItem>
                                <SelectItem value="other">Other</SelectItem>
                              </SelectContent>
                            </Select>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <div className="grid md:grid-cols-2 gap-6">
                        <FormField
                          control={consultationForm.control}
                          name="preferredDate"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Preferred Date</FormLabel>
                              <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                  <SelectTrigger data-testid="select-consultation-date">
                                    <SelectValue placeholder="Select date" />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  <SelectItem value="this-week">This Week</SelectItem>
                                  <SelectItem value="next-week">Next Week</SelectItem>
                                  <SelectItem value="flexible">Flexible</SelectItem>
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={consultationForm.control}
                          name="preferredTime"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Preferred Time</FormLabel>
                              <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                  <SelectTrigger data-testid="select-consultation-time">
                                    <SelectValue placeholder="Select time" />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  <SelectItem value="morning">Morning (9-12)</SelectItem>
                                  <SelectItem value="afternoon">Afternoon (12-17)</SelectItem>
                                  <SelectItem value="evening">Evening (17-18)</SelectItem>
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <FormField
                        control={consultationForm.control}
                        name="notes"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Additional Notes (Optional)</FormLabel>
                            <FormControl>
                              <Textarea
                                {...field}
                                rows={3}
                                placeholder="Any specific topics you'd like to discuss..."
                                data-testid="textarea-consultation-notes"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <Button
                        type="submit"
                        className="w-full bg-accent-500 hover:bg-accent-600 text-white px-8 py-4 font-semibold"
                        disabled={submitConsultation.isPending}
                        data-testid="button-book-consultation"
                      >
                        {submitConsultation.isPending ? "Booking..." : "Book Free Consultation"}
                      </Button>
                    </form>
                  </Form>
                </div>
              )}
            </div>

            {/* Contact Information */}
            <div>
              <h3 className="text-2xl font-bold text-slate-900 mb-8">Contact Information</h3>
              
              <div className="space-y-8 mb-8">
                {contactInfo.map((info, index) => {
                  const IconComponent = info.icon;
                  return (
                    <div
                      key={index}
                      className="flex items-start space-x-4"
                      data-testid={`contact-info-${index}`}
                    >
                      <div className={`w-12 h-12 ${info.color} rounded-full flex items-center justify-center flex-shrink-0`}>
                        <IconComponent className="w-6 h-6" />
                      </div>
                      <div>
                        <h4 className="font-semibold text-slate-900 mb-2">{info.title}</h4>
                        <div className="text-slate-600">{info.content}</div>
                      </div>
                    </div>
                  );
                })}
              </div>

              {/* Free Consultation CTA */}
              <div className="p-6 bg-gradient-to-r from-primary-500 to-accent-500 rounded-xl text-white">
                <h4 className="font-bold text-xl mb-3">Book Your Free Consultation</h4>
                <p className="mb-4 opacity-90">
                  Speak with our healthcare recruitment specialists to discuss your career goals and available opportunities.
                </p>
                <button
                  onClick={() => setActiveForm("consultation")}
                  className="bg-white text-primary-600 hover:bg-slate-100 px-6 py-3 rounded-lg font-semibold transition-colors duration-200"
                  data-testid="button-cta-consultation"
                >
                  Schedule Now
                </button>
              </div>
            </div>
          </div>
        </div>
      </section>
    </>
  );
}
