import { Helmet } from "react-helmet-async";
import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useMutation } from "@tanstack/react-query";
import { z } from "zod";
import { Check, DollarSign, Clock, GraduationCap, Heart, Upload } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import ConsultationCTA from "@/components/ui/consultation-cta";

import nurseimages from "@assets/nurseimages.jpg";

const applicationSchema = z.object({
  firstName: z.string().min(2, "First name must be at least 2 characters"),
  lastName: z.string().min(2, "Last name must be at least 2 characters"),
  email: z.string().email("Invalid email address"),
  phone: z.string().min(10, "Phone number must be at least 10 digits"),
  profession: z.string().min(1, "Please select your profession"),
  experience: z.string().min(1, "Please select your experience level"),
  location: z.string().min(2, "Please enter your preferred location"),
  availability: z.string().min(1, "Please select your availability"),
  message: z.string().min(10, "Please provide more details about your background"),
});

type ApplicationForm = z.infer<typeof applicationSchema>;

export default function JoinTeam() {
  const { toast } = useToast();
  const [isSubmitted, setIsSubmitted] = useState(false);

  const form = useForm<ApplicationForm>({
    resolver: zodResolver(applicationSchema),
    defaultValues: {
      firstName: "",
      lastName: "",
      email: "",
      phone: "",
      profession: "",
      experience: "",
      location: "",
      availability: "",
      message: "",
    },
  });

  const submitApplication = useMutation({
    mutationFn: async (data: ApplicationForm) => {
      return await apiRequest("POST", "/api/contact", {
        ...data,
        inquiryType: "job_application",
      });
    },
    onSuccess: () => {
      setIsSubmitted(true);
      toast({
        title: "Application Submitted",
        description: "We'll review your application and get back to you within 24 hours.",
      });
    },
    onError: () => {
      toast({
        title: "Submission Failed",
        description: "Please try again or contact us directly.",
        variant: "destructive",
      });
    },
  });

  const benefits = [
    {
      icon: DollarSign,
      title: "Competitive Compensation",
      description: "Above-market rates with comprehensive benefits packages tailored to healthcare professionals.",
      color: "bg-primary-100 text-primary-600",
    },
    {
      icon: Clock,
      title: "Flexible Scheduling",
      description: "Choose from permanent, temporary, or contract positions that fit your lifestyle and career goals.",
      color: "bg-accent-100 text-accent-600",
    },
    {
      icon: GraduationCap,
      title: "Professional Development",
      description: "Ongoing training opportunities and career advancement support to help you grow professionally.",
      color: "bg-primary-100 text-primary-600",
    },
    {
      icon: Heart,
      title: "Personal Support",
      description: "Dedicated recruitment consultants who understand your needs and career aspirations.",
      color: "bg-accent-100 text-accent-600",
    },
  ];

  const requirements = [
    "Valid NMC registration (for nurses)",
    "Enhanced DBS check (or willingness to obtain)",
    "Relevant healthcare qualifications",
    "Right to work in the UK",
    "Professional references",
    "Commitment to patient care excellence",
  ];

  if (isSubmitted) {
    return (
      <>
        <Helmet>
          <title>Application Submitted - SmartGad Living</title>
        </Helmet>
        <div className="min-h-screen bg-slate-50 flex items-center justify-center py-20">
          <div className="max-w-md mx-auto text-center bg-white p-8 rounded-xl shadow-lg">
            <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-6">
              <Check className="text-green-600 w-8 h-8" />
            </div>
            <h1 className="text-2xl font-bold text-slate-900 mb-4">Application Submitted!</h1>
            <p className="text-slate-600 mb-6">
              Thank you for your interest in joining SmartGad Living. We'll review your application and contact you within 24 hours.
            </p>
            <Button
              onClick={() => window.location.href = "/"}
              className="bg-primary-600 hover:bg-primary-700"
              data-testid="button-return-home"
            >
              Return to Home
            </Button>
          </div>
        </div>
      </>
    );
  }

  return (
    <>
      <Helmet>
        <title>Join Our Team - SmartGad Living | Healthcare Career Opportunities</title>
        <meta name="description" content="Join SmartGad Living's healthcare team. Competitive compensation, flexible scheduling, and professional development opportunities for nurses and care workers across the UK." />
      </Helmet>
      {/* Hero Section */}
      <section className="relative bg-gradient-to-br from-primary-50 to-slate-100 py-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <h1 className="text-4xl lg:text-5xl font-bold text-slate-900 leading-tight mb-6">
              Join Our Healthcare <span className="text-primary-600">Team</span>
            </h1>
            <p className="text-xl text-slate-600 max-w-3xl mx-auto">
              Take the next step in your healthcare career with opportunities that value your skills, dedication, and commitment to excellent patient care.
            </p>
          </div>
        </div>
      </section>
      {/* Benefits Section */}
      <section className="py-20 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-3xl font-bold text-slate-900 mb-4">Why Choose SmartGad Living?</h2>
            <p className="text-xl text-slate-600">Discover the benefits of building your healthcare career with us</p>
          </div>

          <div className="grid lg:grid-cols-2 gap-16 items-center mb-16">
            <div>
              <img
                src={nurseimages}
                alt="Professional medical team in modern UK hospital setting"
                className="rounded-2xl shadow-2xl w-full h-auto"
              />
            </div>

            <div className="space-y-8">
              {benefits.map((benefit, index) => {
                const IconComponent = benefit.icon;
                return (
                  <div
                    key={index}
                    className="flex items-start space-x-4"
                    data-testid={`benefit-${index}`}
                  >
                    <div className={`w-12 h-12 ${benefit.color} rounded-full flex items-center justify-center flex-shrink-0`}>
                      <IconComponent className="w-6 h-6" />
                    </div>
                    <div>
                      <h3 className="font-semibold text-slate-900 mb-2">{benefit.title}</h3>
                      <p className="text-slate-600">{benefit.description}</p>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>

          {/* Requirements */}
          <div className="bg-slate-50 rounded-xl p-8">
            <h3 className="text-2xl font-bold text-slate-900 mb-6">Basic Requirements</h3>
            <div className="grid md:grid-cols-2 gap-4">
              {requirements.map((requirement, index) => (
                <div
                  key={index}
                  className="flex items-center space-x-3"
                  data-testid={`requirement-${index}`}
                >
                  <Check className="text-primary-600 w-5 h-5 flex-shrink-0" />
                  <span className="text-slate-700">{requirement}</span>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>
      {/* Application Form */}
      <section className="py-20 bg-slate-50">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-slate-900 mb-4">Apply Today</h2>
            <p className="text-xl text-slate-600">
              Ready to start your journey with SmartGad Living? Complete the application form below.
            </p>
          </div>

          <div className="bg-white rounded-xl shadow-lg p-8">
            <Form {...form}>
              <form onSubmit={form.handleSubmit((data) => submitApplication.mutate(data))} className="space-y-6">
                <div className="grid md:grid-cols-2 gap-6">
                  <FormField
                    control={form.control}
                    name="firstName"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>First Name</FormLabel>
                        <FormControl>
                          <Input {...field} data-testid="input-first-name" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="lastName"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Last Name</FormLabel>
                        <FormControl>
                          <Input {...field} data-testid="input-last-name" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <FormField
                    control={form.control}
                    name="email"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Email Address</FormLabel>
                        <FormControl>
                          <Input type="email" {...field} data-testid="input-email" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="phone"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Phone Number</FormLabel>
                        <FormControl>
                          <Input {...field} data-testid="input-phone" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <FormField
                    control={form.control}
                    name="profession"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Professional Role</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={field.value}>
                          <FormControl>
                            <SelectTrigger data-testid="select-profession">
                              <SelectValue placeholder="Select your role" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            <SelectItem value="registered-nurse">Registered Nurse</SelectItem>
                            <SelectItem value="care-worker">Care Worker</SelectItem>
                            <SelectItem value="healthcare-assistant">Healthcare Assistant</SelectItem>
                            <SelectItem value="other">Other</SelectItem>
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="experience"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Experience Level</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={field.value}>
                          <FormControl>
                            <SelectTrigger data-testid="select-experience">
                              <SelectValue placeholder="Years of experience" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            <SelectItem value="0-1">0-1 years</SelectItem>
                            <SelectItem value="2-5">2-5 years</SelectItem>
                            <SelectItem value="6-10">6-10 years</SelectItem>
                            <SelectItem value="10+">10+ years</SelectItem>
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <FormField
                    control={form.control}
                    name="location"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Preferred Location</FormLabel>
                        <FormControl>
                          <Input {...field} placeholder="e.g., Manchester, Birmingham" data-testid="input-location" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="availability"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Availability</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={field.value}>
                          <FormControl>
                            <SelectTrigger data-testid="select-availability">
                              <SelectValue placeholder="When can you start?" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            <SelectItem value="immediately">Immediately</SelectItem>
                            <SelectItem value="1-week">Within 1 week</SelectItem>
                            <SelectItem value="2-weeks">Within 2 weeks</SelectItem>
                            <SelectItem value="1-month">Within 1 month</SelectItem>
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <FormField
                  control={form.control}
                  name="message"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Tell Us About Yourself</FormLabel>
                      <FormControl>
                        <Textarea
                          {...field}
                          rows={4}
                          placeholder="Describe your experience, qualifications, and what type of position you're looking for..."
                          data-testid="textarea-message"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <Button
                  type="submit"
                  className="w-full bg-primary-600 hover:bg-primary-700 text-white px-8 py-4 text-lg font-semibold"
                  disabled={submitApplication.isPending}
                  data-testid="button-submit-application"
                >
                  {submitApplication.isPending ? "Submitting..." : "Submit Application"}
                </Button>
              </form>
            </Form>
          </div>
        </div>
      </section>
      {/* Alternative Contact */}
      <section className="py-20 bg-primary-50">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 className="text-3xl font-bold text-slate-900 mb-6">Prefer to Talk First?</h2>
          <p className="text-xl text-slate-600 mb-8">
            Schedule a free consultation to discuss your career goals and learn more about opportunities available with SmartGad Living.
          </p>
          <ConsultationCTA size="lg" />
        </div>
      </section>
    </>
  );
}
